<?php
session_start();
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/production_errors.log');
error_reporting(E_ALL & ~E_NOTICE & ~E_WARNING);
header('Content-Type: application/json; charset=utf-8');
ob_start();
require('../config.php');

function log_err($msg) {
    error_log("[" . date('Y-m-d H:i:s') . "] " . $msg . "\n", 3, __DIR__ . '/production_errors.log');
}

try {
    // قراءة المدخلات (دعم form-encoded و JSON)
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    $video_id = '';
    $action = '';

    if (stripos($contentType, 'application/json') !== false) {
        $raw = file_get_contents('php://input');
        $data = json_decode($raw, true);
        if (is_array($data)) {
            $video_id = $data['vid'] ?? $data['video_id'] ?? '';
            $action   = $data['action'] ?? '';
        }
    } else {
        $video_id = $_POST['vid'] ?? $_POST['video_id'] ?? '';
        $action   = $_POST['action'] ?? '';
    }

    $video_id = substr(trim($video_id), 0, 64);
    $action = trim($action);

    if (empty($video_id) || !in_array($action, ['like','dislike'], true)) {
        $buf = ob_get_clean();
        if (!empty($buf)) log_err("Pre-output before validation: " . $buf);
        echo json_encode(['success'=>false,'error'=>'Invalid input','details'=>['vid'=>$video_id,'action'=>$action]]);
        exit;
    }

    $_SESSION['liked_videos'] = $_SESSION['liked_videos'] ?? [];
    $_SESSION['disliked_videos'] = $_SESSION['disliked_videos'] ?? [];

    // اتصال DB
    $conn = new mysqli($db_host, $db_user, $db_pass, $db_name);
    if ($conn->connect_error) throw new Exception("DB connect error: " . $conn->connect_error);
    $conn->set_charset('utf8mb4');

    // إدراج/تحديث التصويت (يفترض وجود UNIQUE KEY on (video_id, session_id))
    $session_id = session_id();
    $user_ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';

    $query = "INSERT INTO video_likes (video_id, session_id, action, ip) 
              VALUES (?, ?, ?, ?)
              ON DUPLICATE KEY UPDATE action = VALUES(action), ip = VALUES(ip)";
    $stmt = $conn->prepare($query);
    if (!$stmt) throw new Exception("Prepare failed: " . $conn->error);

    $stmt->bind_param("ssss", $video_id, $session_id, $action, $user_ip);
    if (!$stmt->execute()) throw new Exception("Execute failed: " . $stmt->error);
    $stmt->close();

    // مزامنة الحالة الجلسية
    if ($action === 'like') {
        $_SESSION['disliked_videos'] = array_values(array_diff($_SESSION['disliked_videos'], [$video_id]));
        if (!in_array($video_id, $_SESSION['liked_videos'], true)) $_SESSION['liked_videos'][] = $video_id;
    } else {
        $_SESSION['liked_videos'] = array_values(array_diff($_SESSION['liked_videos'], [$video_id]));
        if (!in_array($video_id, $_SESSION['disliked_videos'], true)) $_SESSION['disliked_videos'][] = $video_id;
    }

    // جلب الإحصائيات بطريقة متوافقة (بدون get_result)
    $stats = $conn->prepare("
        SELECT
            COUNT(CASE WHEN action = 'like' THEN 1 END) AS likes,
            COUNT(CASE WHEN action = 'dislike' THEN 1 END) AS dislikes
        FROM video_likes
        WHERE video_id = ?
    ");
    if (!$stats) throw new Exception("Prepare stats failed: " . $conn->error);
    $stats->bind_param("s", $video_id);
    if (!$stats->execute()) throw new Exception("Execute stats failed: " . $stats->error);

    // استخدام bind_result بدلاً من get_result()
    $stats->store_result();
    $likes = 0; $dislikes = 0;
    $stats->bind_result($likes, $dislikes);
    $stats->fetch();
    $stats->close();

    $response = [
        'success' => true,
        'likes' => (int)$likes,
        'dislikes' => (int)$dislikes,
        'user_action' => $action
    ];

    $buf = ob_get_clean();
    if (!empty($buf)) log_err("Pre-output before sending success: " . $buf);

    echo json_encode($response);
    $conn->close();
    exit;

} catch (Exception $e) {
    log_err("Exception in like-video.php: " . $e->getMessage());
    $buf = ob_get_clean();
    if (!empty($buf)) log_err("Pre-output before exception: " . $buf);
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Server error']);
    exit;
}
?>