(function () {
  "use strict";

  document.addEventListener("DOMContentLoaded", () => {
    initRatingStars();
    initLikeDislike();
  });

  /* -------------------- Helpers -------------------- */

  // Allow overriding the ajax base URL by setting <body data-ajax-base="/myroot/ajax">
  function getAjaxBase() {
    const bodyBase = document.body && document.body.dataset && document.body.dataset.ajaxBase;
    if (bodyBase) return bodyBase.startsWith("http") ? bodyBase : `${window.location.origin}${bodyBase}`;

    // Fallback: take the first non-empty path segment as "root" (safer than split("/")[1])
    const parts = window.location.pathname.split("/").filter(Boolean);
    const root = parts.length ? parts[0] : "";
    return root && root !== "index.php"
      ? `${window.location.origin}/${root}/ajax`
      : `${window.location.origin}/ajax`;
  }

  /* -------------------- 1) RATE STARS -------------------- */
  function initRatingStars() {
    const containers = document.querySelectorAll(".rating-container[data-video-id]");
    if (!containers.length) return;

    const BASE = getAjaxBase();

    containers.forEach((container) => {
      const videoId = container.dataset.videoId;
      // NOTE: some themes/CSS use flex-row-reverse so the DOM order may be reversed.
      // If your markup guarantees left-to-right 1..5 stars, remove `.reverse()`.
      const stars = Array.from(container.querySelectorAll(".star")).reverse();
      const avgEl = document.getElementById(`rating-avg-${videoId}`);
      const cntEl = document.getElementById(`rating-count-${videoId}`);
      let selectedIndex = -1; // visual "average" selection index (0-based)
      let alreadyRated = false; // whether CURRENT user has rated (only set true after successful POST)

      // single toast for all rating containers
      let toast = document.querySelector(".rating-toast");
      if (!toast) {
        toast = document.createElement("div");
        toast.className = "rating-toast";
        toast.textContent = "شكراً لتقييمك!";
        document.body.appendChild(toast);
      }
      function showToast() {
        toast.classList.add("show");
        setTimeout(() => toast.classList.remove("show"), 3000);
      }

      // Render stars: mode = "hover" | "selected" | "none"
      function render(index, mode) {
        stars.forEach((s, i) => {
          s.classList.remove("hover", "selected");
          if (mode === "hover" && i <= index) s.classList.add("hover");
          if (mode === "selected" && i <= index) s.classList.add("selected");
        });
      }

// Load average rating (do NOT assume the user already rated just because an average exists).
fetch(`${BASE}/get_rating.php?video_id=${encodeURIComponent(videoId)}`)
  .then(async (r) => {
    const text = await r.text(); // اقرأ النص أولاً
    const contentType = r.headers.get('content-type') || '';

    if (!r.ok) {
      // غير حالة 200، سجّل النص الكامل لمساعدة التصحيح
      console.error('get_rating non-OK response', r.status, text);
      throw new Error('Server returned status ' + r.status);
    }

    // إذا الـ Content-Type يشير إلى JSON فحاول تحويله
    if (contentType.includes('application/json') || text.trim().startsWith('{') || text.trim().startsWith('[')) {
      try {
        return JSON.parse(text);
      } catch (err) {
        // إذا فشلت JSON.parse سجّل النص الكامل ليتضح سبب المشكلة (مثل صفحة خطأ HTML)
        console.error('get_rating: invalid JSON response:', text);
        throw err;
      }
    } else {
      // الاستجابة ليست JSON على ما يبدو — سجّل النص لمساعدة التصحيح
      console.error('get_rating: unexpected content-type or body:', contentType, text);
      throw new Error('Unexpected response type from server');
    }
  })
  .then((data) => {
    // هنا نضمن أن data هو كائن جافاسكربت صالح
    if (data && data.success) {
      if (avgEl) avgEl.textContent = Number(data.average_rating).toFixed(1);
      if (cntEl) cntEl.textContent = `(${data.total_votes} تقييم)`;
      selectedIndex = Math.round(Number(data.average_rating)) - 1;
      if (selectedIndex >= 0) render(selectedIndex, 'selected');
    } else {
      console.warn('get_rating returned success=false or invalid data', data);
    }
  })
  .catch((err) => {
    // أخطاؤنا (شبكة، JSON غير صالح، أو غير ذلك) تُسجّل هنا
    console.error('get_rating error:', err);
    // اختياري: عرض قيمة افتراضية في الواجهة بدلاً من تركها فارغة
    if (avgEl) avgEl.textContent = '0.0';
    if (cntEl) cntEl.textContent = '(0 تقييم)';
  });

      // Setup each star: mouse and keyboard
      stars.forEach((star, i) => {
        // Make stars focusable & accessible
        star.setAttribute("tabindex", "0");
        star.setAttribute("role", "button");
        star.setAttribute("aria-label", `${i + 1} نجمة`);

        function setHover() {
          if (!alreadyRated) render(i, "hover");
        }
        function clearHover() {
          // If the user hasn't rated, return to showing the rounded average (selectedIndex).
          // If selectedIndex < 0, clear all.
          if (!alreadyRated) {
            if (selectedIndex >= 0) render(selectedIndex, "selected");
            else render(-1, "none");
          } else {
            // user has rated: show their selection
            render(selectedIndex, "selected");
          }
        }

        star.addEventListener("mouseenter", setHover);
        star.addEventListener("mouseleave", clearHover);
        star.addEventListener("focus", setHover);
        star.addEventListener("blur", clearHover);

        // keyboard support (Enter / Space)
        star.addEventListener("keydown", (ev) => {
          if (ev.key === "Enter" || ev.key === " ") {
            ev.preventDefault();
            star.click();
          }
        });

        star.addEventListener("click", () => {
          if (alreadyRated) return;

          const ratingValue = i + 1; // because stars were reversed earlier, i maps to rating
          selectedIndex = i;
          render(i, "selected");

          // Send rating (server should authenticate/validate CSRF as needed)
          fetch(`${BASE}/rate_star.php`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ video_id: videoId, rating: ratingValue }),
          })
            .then((r) => r.json())
            .then((res) => {
              if (!res || !res.success) throw new Error(res ? res.message : "Unknown response");
              alreadyRated = true;
              showToast();
              // Refresh shown average and count
              return fetch(`${BASE}/get_rating.php?video_id=${encodeURIComponent(videoId)}`);
            })
            .then((r) => r.json())
            .then((data) => {
              if (data && data.success) {
                if (avgEl) avgEl.textContent = Number(data.average_rating).toFixed(1);
                if (cntEl) cntEl.textContent = `(${data.total_votes} تقييم)`;
              }
            })
            .catch((err) => {
              console.error("rating error", err);
              // revert visual selection if desired (optional)
              if (!alreadyRated && selectedIndex >= 0) {
                // show average again
                if (selectedIndex >= 0) render(selectedIndex, "selected");
              }
            });
        });
      });
    });
  }

  /* -------------------- 2) LIKE / DISLIKE -------------------- */
  function initLikeDislike() {
    const boxes = document.querySelectorAll(".video-tool-box[data-video-id]");
    if (!boxes.length) return;

    const BASE = getAjaxBase();

    boxes.forEach((box) => {
      const vid = box.dataset.videoId;
      const likeBtn = box.querySelector(".like-button");
      const dislikeBtn = box.querySelector(".dislike-button");
      const likeCountEl = box.querySelector(`#like-count-${vid}`);
      const dislikeCountEl = box.querySelector(`#dislike-count-${vid}`);

      function updateButtons(data) {
        if (!data) return;
        if (likeCountEl) likeCountEl.textContent = data.likes;
        if (dislikeCountEl) dislikeCountEl.textContent = data.dislikes;
        if (likeBtn) likeBtn.classList.toggle("active", data.user_action === "like");
        if (dislikeBtn) dislikeBtn.classList.toggle("active", data.user_action === "dislike");
      }

      // load current state (this endpoint seems to return user_action as well)
      fetch(`${BASE}/get_votes_with_user.php?vid=${encodeURIComponent(vid)}`)
        .then((r) => r.json())
        .then((data) => {
          if (data && data.success) updateButtons(data);
        })
        .catch((err) => console.error("fetch votes error:", err));

      if (likeBtn) likeBtn.addEventListener("click", () => send("like"));
      if (dislikeBtn) dislikeBtn.addEventListener("click", () => send("dislike"));

      function send(action) {
        fetch(`${BASE}/like-video.php`, {
          method: "POST",
          headers: {
            "Content-Type": "application/x-www-form-urlencoded",
          },
          body: `vid=${encodeURIComponent(vid)}&action=${encodeURIComponent(action)}`,
        })
          .then((r) => r.json())
          .then((data) => {
            if (data && data.success) updateButtons(data);
            else console.error(`send ${action} failed`, data);
          })
          .catch((err) => console.error(`send ${action} error:`, err));
      }
    });
  }
})();